Scriptname RF:TravelHandlerQuest Extends Quest

Actor PlayerRef
SpaceshipReference MyShip

RF:FuelHandlerQuest Property FuelManager Auto Mandatory Const
RF:TutorialQuest Property TutorialManager Mandatory Const Auto
SQ_PlayerShipScript property SQ_PlayerShip Auto Const
GlobalVariable Property _RF_Sys_AllowTravel Mandatory Const Auto
{ Travel where 1 is easy mode }

Group Globals 
    GlobalVariable Property _RF_DistanceRestriction Mandatory Const Auto
    { Severity tied to SPEL record. 0:None 1:40% 2:60% 3:90% (should be functionally inter-system only) }
    GlobalVariable Property _RF_GravDebuff Mandatory Const Auto
    { 1: 2x 2: 4x 3:8x }
    GlobalVariable Property _RF_TRV_IsUsingEmergencyJump Mandatory Const Auto
    { This should be set in CheckRestrictions() to trigger our coinflip to damage stuff }
    GlobalVariable Property _RF_Sys_TravelLimitHardcore Mandatory Const Auto
EndGroup

Group Limiter
    MiscObject Property _RF_TravelKill Auto Mandatory Const
EndGroup

Group Messages
    Message Property _RF_Alert_GravJumpDisabled Mandatory Const Auto
    Message Property _RF_Alert_ShipDamage_Overheat Mandatory Const Auto
    Message Property _RF_Alert_GravJumpEnabled Mandatory Const Auto
    Message Property _RF_Alert_Map_ShipDamage Mandatory Const Auto
    Message Property _RF_Alert_ShipDamage Mandatory Const Auto
    Message Property _RF_Alert_TooManyJumps Mandatory Const Auto
EndGroup

Group Conditions
    Keyword Property LocTypeSENotAllowed Mandatory Const Auto
    ConditionForm Property _RF_InCooldownState Mandatory Const Auto
    { Indicates we are cooling down from a jump }
    ConditionForm Property _RF_Overheated Mandatory Const Auto
    { Indicates Overheat spell is still running - prevent re-enable from other modules }
EndGroup

Group Spells
    Spell Property _RF_GravDriveDetectiveSpell Mandatory Const Auto
    { This is our cooldown, I think}
    Spell Property _RF_GravDriveOverheatSpell Mandatory Const Auto
    { This disables travel on start and enables it on end if we meet condtions in this quest}
EndGroup

Group ActorValues
    ActorValue Property SpaceshipGravDriveHealth Mandatory Const Auto
EndGroup


bool GravJumpOK = true ; This is set by overheat or damage checks
bool FarTravelOK = true ; This is set by catastrophic events we are not using yet

;Counts for breakpoints
int BreakHigh = 70
int BreakNorm = 40
int BreakMerg = 25
int BreakCrit = 10

int LastPercent = 0

;Used for exceptions or out-of-fuel scenarios
bool UsingEmergencyJump = false ; Configurable... somewhere
bool WasGravJumpDisabled = false ; Updated in Restrictions() to avoid message spam
bool ShipInWastes = false ; Gets set in CheckRestrictions() to trigger our fallback travel limiters
bool Hardcore = false ; This is our contraband limiter

int SequentialFastJumps = 0 ; Set in whatever handler it is, not used yet

InputEnableLayer FarTravelInputLayerTHQ
InputEnableLayer GravJumpInputLayerTHQ

;System
bool Running = false ; Is the mod running
bool DEBUGON = true ; Debug handler
bool DEBUGNOTIFY = false ; I can't tell how fast the scripts are running with this spamming me lol
bool DEBUGFORCE = false ; This makes coinflips always return true. DISABLE FOR PROD

;----------------------------------------------------
;----------------- DEBUG AND SYSTEM ----------------- 
;----------------------------------------------------


bool Function RollDice(int aiWeight = 50)
    If DEBUGFORCE
        Return True
    Else
        Return FuelManager.CoinFlip(aiWeight)
    EndIF
EndFunction

;Generic debug function that can be disabled for prod
Function DBG(String asTextToPrint = "Debug Error!")
    If DEBUGON
        Debug.Trace("RF TravelHandler: " + asTextToPrint)
        IF DEBUGNOTIFY ; Can log in realtime to avoid alt-tabbing constantly
            Debug.Notification("RTRAV: " + asTextToPrint)
        EndIF
    EndIF
EndFunction

Function DBGVB(String asTextToPrint = "Debug Error!")
    If DEBUGON
        Debug.Notification("RTRAV: " + asTextToPrint)
    EndIF
EndFunction

; True indicates all elements of Travel are running
bool Function TravelSystemEnabled()
    Running = True
    IF _RF_Sys_TravelLimitHardcore.GetValue() == 1
        Hardcore = True
    Else
        Hardcore = False
        Limit(0)
    EndIF
    if _RF_Sys_AllowTravel.GetValue() == 1
        DBG("TSE killing process with " + Running + " from " + _RF_Sys_AllowTravel.GetValue())
        If Running
            Debug.Notification("Travel restrictions removed.")
        EndIf
        Running = false
        CleanUpRestrictions()
        HARDCORE = False
    Else
        If !FuelManager.ModRunning()
            Debug.Notification("Real Fuel Travel system shutting down.")
            CleanUpRestrictions()
            Running = False
            Hardcore = False
        Else ; This means Travel = 0 and ModRunning
            If !Running
                Debug.Notification("Travel restrictions enabled.")
            EndIF
            Running = True
        EndIf
    EndIf
    return Running
EndFunction

bool Function TravelSystemRunning()
    Return Running
EndFunction

Function CleanUpRestrictions()
    DisableFar(false)
    DisableGrav(false)
    UsingEmergencyJump = False
    Limit(0)
EndFunction

Function Limit(bool abLimit = true)
    If !abLimit
        If PlayerRef.GetItemCount(_RF_TravelKill) >= 1
            PlayerRef.RemoveItem(_RF_TravelKill, 999, true)
        EndIf
    Else
        PlayerRef.AddItem(_RF_TravelKill, 1, true)
    EndIF
EndFunction

; True = No Grav Jumping
; False = Grav Jumping Is Enabled
Function DisableGrav(bool abDisabling)
    If abDisabling
        If (GravJumpInputLayerTHQ == None)
            GravJumpInputLayerTHQ = InputEnableLayer.Create()
            GravJumpInputLayerTHQ.EnableGravJump(False)
            GravJumpOK = False
            DBG("Disabling Grav Jump")
        Else
            DBG("EXCEPTION - DisableGrav called with no active input layer present!")
        EndIF
    Else
        If (GravJumpInputLayerTHQ != None)
            GravJumpInputLayerTHQ.EnableGravJump(True)
            GravJumpInputLayerTHQ.Delete()
            GravJumpInputLayerTHQ = None
            GravJumpOK = True
            DBG("Enabling Grav Jump")
        EndIf
    EndIF
EndFunction

; True = No Far Travel
; False = Far Travel Enabled
Function DisableFar(bool abDisabling)
    If abDisabling
        If (FarTravelInputLayerTHQ == None)
            FarTravelInputLayerTHQ = InputEnableLayer.Create()
            FarTravelInputLayerTHQ.EnableFarTravel(False)
            FarTravelOK = False
            DBG("Disabling Far Travel")
        Else
            DBG("EXCEPTION - DisableFar called with no active input layer present!")
        EndIf
    Else
        If (FarTravelInputLayerTHQ != None)
            FarTravelInputLayerTHQ.EnableFarTravel(True)
            FarTravelInputLayerTHQ.Delete()
            FarTravelInputLayerTHQ = None
            FarTravelOK = True
            DBG("Enabling Far Travel")
        EndIf
    EndIF
EndFunction

Function EnableAndNotify()
    If (GravJumpInputLayerTHQ != None)
        If !GetOverheated()
            DisableGrav(false)
            If WasGravJumpDisabled
                TutorialManager.HelpMessage(_RF_Alert_GravJumpEnabled)
            EndIf
            WasGravJumpDisabled = False
        Else
            Debug.Notification("Grav Drive overheated! Repair to continue travel.")
            DBG("EnableAndNotify called while overheated!!!")
        EndIF
    EndIF
EndFunction

Function DisableAndNotify(bool abNotify = true)
    bool Notify = false
    If (GravJumpInputLayerTHQ == None)
        DisableGrav(true)
        WasGravJumpDisabled = True
        TutorialManager.Nag(0)
    EndIF
EndFunction

;Mostly for curiosity
float Function GetGravRange()
    float Range = 1.0
    Range = MyShip.GetGravJumpRange()
    If (GravJumpInputLayerTHQ != None)
        DBG("GetGravRange thinks Grav Jump is disabled")
    Else
        DBG("GetGravJump thinks our range is" + Range)
    EndIf
    Return Range
EndFunction

;Check this in EnableAndNotify to avoid Dashboard triggering an early reset and confusing the player
bool Function GetOverheated()
    Return _RF_Overheated.IsTrue(PlayerRef)
EndFunction

; This simply shuts off fast travel back to hubs. 
; Obviously it allows travel back to non-hub land locations, but whatever
Function SmartLimitHandler(int aiType)
    If Hardcore
        Location PlayerIsIn = MyShip.GetCurrentLocation()
        If FuelManager.IsSettled(PlayerIsIn)
            Limit(0)
            DBG("SmartLimitHandler cleared with Settled")
        Else
            IF aiType > 1
                Limit()
                DBG("SmartLimitHandler active with Type " + aiType )
            EndIf
        EndIf
    Else
        DBG("Limits disabled")
    EndIf
EndFunction

; Pretty simple actually
; Not called yet. I may use the same event as emergencyjump with a different message...
Function CooldownHandler()
    If Running
        If _RF_InCooldownState.IsTrue(PlayerRef)
            SequentialFastJumps += 1
            DBG("CooldownHandler called with InCooldownState True" + SequentialFastJumps)
        Else
            SequentialFastJumps = 0
            DBG("CooldownHandler called with InCooldownState False. Clearing FastJumps " + SequentialFastJumps)
        EndIf
        If SequentialFastJumps > 4 || DEBUGFORCE
            IF RollDice() || DEBUGFORCE
                DBG("Firing overheat spell...")
                _RF_GravDriveOverheatSpell.Cast(PlayerRef)
            EndIF
        EndIF
    EndIf
EndFunction


;-------------------------------------------------
;----------------- TRAVEL SYSTEM -----------------
;-------------------------------------------------


; This is the entire core of our travel system disabler.
; It operates purely on fuel percentage to limit mobility.
Function CheckRestrictions(int aiPercent = 120)
    If Running
        UsingEmergencyJump = False
        Bool Disable = False
        If aiPercent > 100
            aiPercent = FuelManager.GetFuelLevelInt()
        EndIf
        int Restriction = 0
        float Mult = 1.0 ; Skill scaling TBI
        If aiPercent > ( BreakHigh * Mult) ; Ease any restrictions present
            Restriction = 0
        ElseIf aiPercent > ( BreakNorm * Mult ) ; 80% range
            Restriction = 1
        ElseIf aiPercent > ( BreakMerg * Mult ) ; 70% range
            Restriction = 2
        ElseIf aiPercent > BreakCrit ; 60% range
            Restriction = 3
            UsingEmergencyJump = True
            TutorialManager.ShowTutorial(1)
        Elseif aiPercent <= BreakCrit ; 0 range if hardcore - nothing over Merg otherwise
            Restriction = 3
            DBG("Restriction 3 on Critical")
            Disable = true
            TutorialManager.ShowTutorial(2)
        EndIf
        If Disable
            DisableAndNotify() ; This just keeps the system alive and ticks up the tutorial counter
        Else
            EnableAndNotify()
        EndIf
        _RF_DistanceRestriction.SetValue(Restriction) ; This alters our grav drive max range to: 100%, 50%, 35%, and of course at 3 it is disabled entirely.
        _RF_GravDebuff.SetValue(Restriction) ; This hampers our Grav Drive spool up speed. Conveniently, it uses the same values
        DBG("CheckRestrictions: Tier" + Restriction + " from " + aiPercent + "% fuel.")
    Else
        int Restriction = 0
        _RF_DistanceRestriction.SetValue(Restriction) ; This alters our grav drive max range to: 100%, 50%, 35%, and of course at 3 it is disabled entirely.
        _RF_GravDebuff.SetValue(Restriction) ; This hampers our Grav Drive spool up speed. Conveniently, it uses the same values
        UsingEmergencyJump = False
        DisableGrav(false)
        DisableFar(false)
    EndIF
EndFunction

; Reduce an AV by X% of its current amount to avoid going negative if wanted
Function DamageValueSafe(ActorValue ValueToDamage, float afAmountToDamage = 0.75)
    float BaseAV = MyShip.GetBaseValue(ValueToDamage)
    float CurrAV = MyShip.GetValue(ValueToDamage)
    float CurrCent = CurrAV / BaseAV
    DBG("DamageValueSafe got base " + BaseAV + " current " + CurrAV + " ratio " + CurrCent)
    float DamageToDo = CurrAV * afAmountToDamage
    MyShip.DamageValue(ValueToDamage, DamageToDo)
    DBG("DamageValueSafe damaged" + ValueToDamage + " by " + DamageToDo + " resulting " + MyShip.GetValue(ValueToDamage) ) 
EndFunction

; If this fires we want to skip all fueling logic except to blow out what's left of our tanks. 
; We should also not fire ANY notif except this one - spell now has no notif on start for versatility
bool Function HandleEmergencyJump()
    DBG("HandleEmergencyJump firing with UsingEmergency " + UsingEmergencyJump)
    bool DidDamageOccur = false ; If this happens we force drain fuel as well.
    If UsingEmergencyJump || _RF_TRV_IsUsingEmergencyJump.GetValue() == 1 ; Global is PURELY FOR DEBUG!
        ActorValue Hull = Game.GetHealthAV()
        DidDamageOccur = RollDice()
        If DidDamageOccur
            DBG("EmergencyJump failure mode fired and damaged Grav Drive")
            float Max = MyShip.GetBaseValue(Hull) * 0.45
            float Damage = Max * FuelManager.GetVariance(0.2)
            Utility.Wait(0.1)
            MyShip.DamageValue(Hull, Damage) ; This might kill you - cool!
            float GD = MyShip.GetValue(SpaceshipGravDriveHealth)
            TutorialManager.HelpMessage(_RF_Alert_ShipDamage_Overheat)
            ;DamageValueSafe(SpaceshipGravDriveHealth, 4.75) ; This was a bit egregious but cool
            DBG("Damaged Grav Drive for " + Damage + " resulting " + MyShip.GetValue(SpaceshipGravDriveHealth) + " from "  + GD)  
            _RF_GravDriveOverheatSpell.Cast(PlayerRef)
            UsingEmergencyJump = False
        EndIf
    EndIf
    Return DidDamageOccur
EndFunction

bool Function HandleOverheatEnd()
    bool OK = false
    float HP = MyShip.GetValue(SpaceshipGravDriveHealth)
    If HP > 1 || !Running
        OK = True
    EndIF
    DBG("HandleOverheatEnd found our Grav Drive health at" + HP + " for " + OK)
    Return OK
EndFunction

bool Function CheckTravelStatus()
    Return GravJumpOK && FarTravelOK ; If either of these is false we skip notifs
EndFunction

Event OnQuestInit()
    StartTimer(7)
EndEvent

Event OnTimer(int aiTimerID)
    PlayerRef = Game.GetPlayer()
    MyShip = FuelManager.GetShip("TravelManager")
EndEvent

; If this returns true that means this jump failed explosively. We probably died
bool Function HandleTravelRestriction(Location akOldLoc, Location akNewLoc)
    bool OKGO = true
    If Running
        MyShip = FuelManager.GetShip("TravelManager")
        OKGO = HandleEmergencyJump()
    Else
        CleanUpRestrictions()
    EndIF
    If !FuelManager.ModRunning()
        CleanUpRestrictions()
        Running = False
    EndIf
    Return OKGO
EndFunction